/** BPkit: Branch Prediction Toolkit          Kenji KISE, Tokyo Tech **/
/**********************************************************************/
#include <zlib.h>
#include "define.h"

#define BUFSIZE 8192
#define TRACE_OFFSET 22 /* do not change the value */

#define VER "BPkit: Branch Prediction Toolkit Ver 0.5.5 (2007-04-17)"
/**********************************************************************/
extern int bp_init(FILE*, int, int*);            /* initialize      */
extern int bp_finalize(int*);                    /* finalize        */
extern int bp_predict(unsigned int, int*, int*); /* predict         */
extern int bp_regist(unsigned int, int, int*);   /* update, regist  */
extern int bp_name(int*);                        /* print your name */

/**********************************************************************/
static char UsageMessage[] =  "\
 -e[num]: end predicton at num branch evaluation.\n\
 -o: generate an output log file.\n\
 -t: print miss rate at 2000 branch interval.\n\
";

int end_count = 0;
int interval_output = 0;

FILE *fp = stdout;
char trace_name[BUFSIZE];

/** function to return the time in nanosec                           **/
/**********************************************************************/
long long get_time()
{
    struct timeval  tp;
    gettimeofday(&tp, NULL);
    return (tp.tv_sec * 1000000ull + tp.tv_usec);
}

/**********************************************************************/
void parse_option(char **option)
{
    if(option==NULL) return;
    for(int i=0; option[i]!=NULL; i++){
        char opt[1024];
        strcpy(opt, option[i]);
        if(opt[0]!='-') continue;
        switch(opt[1]){
        case 'o': 
            char fname[1024];
            strncat(fname, trace_name, strlen(trace_name)-1);
            strcat(fname, ".log");

            if((fp = fopen(fname, "a+"))==NULL){
                printf("fopen error in parse_option\n");
            }
            printf("logfile: %s\n", fname);
            break;
        case 'e': 
            end_count = atoi(&opt[2]);
            break;
        case 't': 
            interval_output = 1;
            break;
        }
    }
}

/**********************************************************************/
int main(int argc, char **argv){
    int count=0, hit=0, mis=0;
    int pc, pred, taken;
    int n, inst;
    long long usec_start, usec_end;
    gzFile gzfp;
    char buf[BUFSIZE];
    long long usec = get_time();
    long long usec2 = usec;

    if(argc==1){
        printf("%s\n", VER);
        printf("usage: %s [options] trace_filename\n", argv[0]);
        printf("%s", UsageMessage);
        exit(0);
    }
    if((gzfp = gzopen(argv[argc-1], "r"))==NULL){
        printf("Bad file name: %s\n", argv[1]);
        exit(0);
    }
    gzgets(gzfp, buf, BUFSIZE);
    if(strncmp(buf, "/***** BPKit 0.5 trace", 20)){
        printf("%s: not a BPKit trace file!\n", argv[1]);
        exit(1);
    }

    gzgets(gzfp, buf, BUFSIZE);
    strcat(trace_name, buf+TRACE_OFFSET);
    
    gzgets(gzfp, buf, BUFSIZE);
    sscanf(buf+TRACE_OFFSET, "%d", &n);

    gzgets(gzfp, buf, BUFSIZE);
    sscanf(buf+TRACE_OFFSET, "%d", &inst);

    parse_option(argv);

    bp_init(fp, n, NULL); /* initialize a branch predictor */

    fprintf(fp, "+ %s\n", VER);
    fprintf(fp, "+ BPKit: ");
    bp_name(NULL);

    usec_start = get_time();
    while(!gzeof(gzfp)){
        gzgets(gzfp, buf, BUFSIZE);
        sscanf(buf, "%x %d", &pc, &taken);

        bp_predict(pc, NULL, &pred); /* prediction */
        bp_regist(pc, taken, NULL);  /* update storage */

        if(pred==taken) hit++; else mis++;        
        count++;
        if(end_count!=0 && end_count<=count) break;

        if(interval_output && (count>0 && count%2000==0)){
            fprintf(fp, "[%08d] ",  count);
            fprintf(fp, "%8.2f ", (get_time() - usec )/1000000.0);
            fprintf(fp, "%6.2f :",(get_time() - usec2)/1000000.0);
            fprintf(fp, "%8d %8d %7d ", count, hit, mis);
            fprintf(fp, ":%8.3f\n", (float)mis/(float)(hit+mis) * 100.0);
            usec2 = get_time();
            fflush(fp);
        }

    }
    usec_end = get_time();

    bp_finalize(NULL); /* finalize a branch predictor */

    fprintf(fp, "+ Elapsed time(sec) %12.5f\n",
            (float)(usec_end - usec_start)/1000000.0);
    fprintf(fp, "#%11d %10d : %10d %8d : ", inst, count, hit, mis);
    fprintf(fp, "%8.4f ", 100.0 - (float)hit/(float)(hit+mis)*100.0);
    if(n==(hit+mis))
        fprintf(fp, "%8.4f ", (float)mis/(float)inst*1000.0);
    else fprintf(fp, " ------- ");
    fprintf(fp, ": %s", trace_name);
    
    gzclose(gzfp);
    return 0;
}
/**********************************************************************/
